this.markmap = this.markmap || {};
(function (exports) {
  'use strict';

  function _extends() {
    _extends = Object.assign || function (target) {
      for (var i = 1; i < arguments.length; i++) {
        var source = arguments[i];
        for (var key in source) {
          if (Object.prototype.hasOwnProperty.call(source, key)) {
            target[key] = source[key]
          }
        }
      }
      return target
    };
    return _extends.apply(this, arguments)
  }

  Math.random().toString(36).slice(2, 8);

  function memoize(fn) {
    const cache = {};
    return function memoized(...args) {
      const key = `${args[0]}`;
      let data = cache[key];
      if (!data) {
        data = { value: fn(...args) };
        cache[key] = data
      }
      return data.value
    }
  }

  function createElement(tagName, props, attrs) {
    const el = document.createElement(tagName);
    if (props) {
      Object.entries(props).forEach(([key, value]) => {
        el[key] = value
      })
    }
    if (attrs) {
      Object.entries(attrs).forEach(([key, value]) => {
        el.setAttribute(key, value)
      })
    }
    return el
  }

  const memoizedPreloadJS = memoize(url => {
    document.head.append(createElement('link', { rel: 'preload', as: 'script', href: url }))
  });

  function loadJSItem(item, context) {
    if (item.type === 'script') {
      return new Promise((resolve, reject) => {
        var _item$data;
        document.head.append(createElement('script', _extends({}, item.data, { onload: resolve, onerror: reject })));
        if (!((_item$data = item.data) != null && _item$data.src)) resolve()
      })
    }
    if (item.type === 'iife') {
      const { fn, getParams } = item.data;
      fn(...((getParams == null ? void 0 : getParams(context)) || []))
    }
  }

  function loadCSSItem(item) {
    if (item.type === 'style') {
      document.head.append(createElement('style', { textContent: item.data }))
    } else if (item.type === 'stylesheet') {
      document.head.append(createElement('link', _extends({ rel: 'stylesheet' }, item.data)))
    }
  }

  async function loadJS(items, context) {
    const needPreload = items.filter(item => {
      var _item$data2;
      return item.type === 'script' && ((_item$data2 = item.data) == null ? void 0 : _item$data2.src)
    });
    if (needPreload.length > 1) needPreload.forEach(item => memoizedPreloadJS(item.data.src));
    context = _extends({ getMarkmap: () => window.markmap }, context);
    for (const item of items) {
      await loadJSItem(item, context)
    }
  }

  function loadCSS(items) {
    for (const item of items) {
      loadCSSItem(item)
    }
  }

  var _window$markmap, _window$markmap$autoL, _window$markmap2, _window$markmap2$auto;
  const enabled = {};
  const ready = loadJS(((_window$markmap = window.markmap) == null ? void 0 : (_window$markmap$autoL = _window$markmap.autoLoader) == null ? void 0 : _window$markmap$autoL.baseJs) || [{
    type: 'script',
    data: { src: 'js/d3.min.js' }
  }, { type: 'script', data: { src: 'js/markmap-view.min.js' } }]).then(() => {
    var _markmap$autoLoader;
    const { markmap } = window;
    loadCSS([{ type: 'style', data: markmap.globalCSS }]);

    // Add custom styles for cleaner appearance and vibrant colors
    loadCSS([{
      type: 'style',
      data: `
        .markmap-node-circle {
          stroke-width: 2px !important;
          r: 6 !important;
        }
        .markmap-node text {
          text-shadow: none !important;
          font-weight: 500 !important;
          font-size: 16px !important;
        }
        .markmap svg {
          filter: none !important;
          box-shadow: none !important;
          border: none !important;
          border-radius: 8px !important;
          background: rgba(0, 0, 0, 0.03) !important;
          margin: 0 !important;
          padding: 0 !important;
          box-sizing: border-box !important;
          width: 100% !important;
          height: auto;
          max-height: none !important;
          overflow: visible !important;
        }
        @media (prefers-color-scheme: dark) {
          .markmap svg {
            background: rgba(255, 255, 255, 0.03) !important;
          }
        }
        .markmap {
          margin: 0 !important;
          padding: 0 !important;
          background: transparent !important;
          width: 100% !important;
          height: var(--markmap-height, 480px);
          min-height: var(--markmap-height, 480px);
          max-height: none !important;
          overflow: visible !important;
          position: relative !important;
          display: block !important;
          visibility: visible !important;
        }
        .markmap-link {
          stroke-width: 2px !important;
        }
        .markmap-foreign code {
          background-color: transparent !important;
          border-radius: 8px !important;
        }
        .language-markmap {
          background: transparent !important;
        }
        pre.markmap-image-container {
          background: transparent !important;
          padding: 0 !important;
        }
        .markmap-node {
          cursor: pointer !important;
        }
        .markmap-node > g > text {
          fill: currentColor !important;
          font-family: inherit !important;
        }
        .markmap g[data-depth] {
          visibility: visible !important;
        }
      `
    }]);
    (_markmap$autoLoader = markmap.autoLoader) == null ? void 0 : _markmap$autoLoader.onReady == null ? void 0 : _markmap$autoLoader.onReady()
  });

  function transform(transformer, content) {
    const { root, features } = transformer.transform(content);
    const keys = Object.keys(features).filter(key => !enabled[key]);
    keys.forEach(key => {
      enabled[key] = true
    });
    const { styles, scripts } = transformer.getAssets(keys);
    const { markmap } = window;
    if (styles) markmap.loadCSS(styles);
    if (scripts) markmap.loadJS(scripts);
    return root
  }

  function render(el) {
    try {
      // Ensure markmap is available
      if (!window.markmap || !window.markmap.Markmap || !window.markmap.Transformer) {
        console.warn('Markmap not ready, retrying...');
        setTimeout(() => render(el), 200);
        return;
      }

      const { Transformer, Markmap, autoLoader } = window.markmap;
      const lines = el.textContent.split('\n');
      let indent = Infinity;
      lines.forEach(line => {
        const spaces = line.match(/^\s*/)[0].length;
        if (spaces < line.length) indent = Math.min(indent, spaces)
      });
      const content = lines.map(line => line.slice(indent)).join('\n');

      // Validate content
      if (!content.trim()) {
        console.warn('Empty markmap content');
        return;
      }

      const transformer = new Transformer(autoLoader == null ? void 0 : autoLoader.transformPlugins);

    // Calculate dynamic height based on content structure
    const nonEmptyLines = lines.filter(line => line.trim().length > 0);
    const nodeCount = nonEmptyLines.length;
    // Calculate tree depth by analyzing indentation levels
    const depths = nonEmptyLines.map(line => {
      const match = line.match(/^(\s*)/);
      return match ? Math.floor(match[1].length / 2) : 0;
    });
    const maxDepth = Math.max(...depths, 1);

    // More generous height calculation: base + nodes + depth factor
    const baseHeight = 300;
    const nodeHeight = nodeCount * 30;
    const depthFactor = maxDepth * 56;
    const dynamicHeight = Math.max(baseHeight, nodeHeight + depthFactor);

    el.style.setProperty('--markmap-height', `${dynamicHeight}px`);
    el.style.height = `${dynamicHeight}px`;
    el.style.minHeight = `${dynamicHeight}px`;

    el.innerHTML = '<svg></svg>';
    const svg = el.firstChild;

    // Ensure SVG is properly initialized
    if (!svg) {
      console.error('Failed to create SVG element');
      return;
    }

    svg.style.height = `${dynamicHeight}px`;
    svg.style.minHeight = `${dynamicHeight}px`;

    // Zoom state management removed - using autoFit instead

    let mm;
    try {
      mm = Markmap.create(svg, {
      embedGlobalCSS: true,
      duration: 300,
      autoFit: true,
      paddingX: 16,
      paddingY: 16,
      fitRatio: 0.92,
      spacingVertical: 20,
      spacingHorizontal: 20,
      zoom: true,
      pan: false,
      font: 'TsangerJinKai02-W04, -apple-system, BlinkMacSystemFont, Helvetica Neue, PingFang SC, Hiragino Sans GB, Microsoft YaHei UI, Microsoft YaHei, Arial, sans-serif',
      color: (d) => {
        // Vibrant color palette
        const colors = [
          '#FF6B6B', // Coral Red
          '#4ECDC4', // Turquoise
          '#45B7D1', // Sky Blue
          '#96CEB4', // Mint Green
          '#FFEAA7', // Warm Yellow
          '#DDA0DD', // Plum
          '#FFA07A', // Light Salmon
          '#20B2AA', // Light Sea Green
          '#87CEEB', // Sky Blue
          '#DEB887', // Burlywood
          '#FF69B4', // Hot Pink
          '#32CD32'  // Lime Green
        ];
        return colors[d.depth % colors.length];
      }
    });
    } catch (error) {
      console.error('Failed to create Markmap instance:', error);
      return;
    }

    if (!mm) {
      console.error('Markmap instance is null');
      return;
    }

    // Zoom and pan disabled - no event listeners needed

    const doRender = (retryCount = 0) => {
      try {
        const root = transform(transformer, content);
        if (!root) {
          console.warn('Transform failed, no root data');
          if (retryCount < 2) {
            setTimeout(() => doRender(retryCount + 1), 100);
          }
          return;
        }

        // Validate root structure
        if (typeof root !== 'object') {
          console.warn('Invalid root type:', typeof root);
          return;
        }

        try {
          mm.setData(root);

          // Multiple retry mechanism for fit
          const attemptFit = (attempts = 0) => {
            try {
              if (mm && typeof mm.fit === 'function') {
                mm.fit();
              }
            } catch (e) {
              if (attempts < 3) {
                console.warn(`Markmap fit failed (attempt ${attempts + 1}), retrying...`, e);
                setTimeout(() => attemptFit(attempts + 1), 200);
              } else {
                console.error('Markmap fit failed after 3 attempts:', e);
              }
            }
          };

          // Delay fit to ensure DOM is ready
          setTimeout(() => attemptFit(), 100);

        } catch (error) {
          console.error('Failed to set markmap data:', error);
          if (retryCount < 2) {
            setTimeout(() => doRender(retryCount + 1), 300);
          }
        }
      } catch (error) {
        console.error('Markmap render failed:', error);
        if (retryCount < 2) {
          setTimeout(() => doRender(retryCount + 1), 500);
        }
      }
    };

    transformer.hooks.retransform.tap(() => doRender());
    doRender();

    } catch (outerError) {
      console.error('Markmap render function failed:', outerError);
    }
  }

  async function renderAllUnder(container) {
    await ready;
    container.querySelectorAll('.markmap').forEach(render)
  }

  function renderAll() {
    return renderAllUnder(document)
  }

  if (!((_window$markmap2 = window.markmap) != null && (_window$markmap2$auto = _window$markmap2.autoLoader) != null && _window$markmap2$auto.manual)) {
    if (document.readyState === 'loading') document.addEventListener('DOMContentLoaded', () => {
      renderAll()
    }); else renderAll()
  }
  exports.ready = ready;
  exports.render = render;
  exports.renderAll = renderAll;
  exports.renderAllUnder = renderAllUnder
})(this.markmap.autoLoader = this.markmap.autoLoader || {});
